using System;
using System.Drawing;
using DynamicNode.Core;
using DynamicNode.Ext.Visual;
using System.Windows.Forms;
using System.Drawing.Drawing2D;

namespace DynamicNode.Examples
{
    /// <summary>
    /// <para>
    /// Beispiel-Implementierung eines benutzerdefinierten Knotens.
    /// </para>
    /// <para>
    /// Dieser Knoten fhrt eine Addition von zwei Int32-Werten durch,
    /// passt die Visualisierung des Knotens an,
    /// stellt einen ViewPort zur Verfgung auf dem die Ergebnisse 
    /// protokolliert werden und zeigt ein
    /// Steuerelement in der Ansicht fr die Knotensteuerung an.
    /// Um die Werte auf den Ein- und Ausgngen zu speichern wird der
    /// Persistenz-Mechanismus der Knoten benutzt.
    /// </para>
    /// </summary>
    [StaticNodeInfo(typeof(NSampleNode), "GetNodeInfo")]
    public class NSampleNode : Node
    {
        /// <summary>
        /// Stellt statisch die Meta-Daten des Knotens zur Verfgung.
        /// Diese Methode wird von dem Attribut <see cref="StaticNodeInfoAttribute"/>
        /// aufgerufen.
        /// </summary>
        /// <returns>Die Knoten-Info.</returns>
        public static NodeInfoAttribute GetNodeInfo()
        {
            return new NodeInfoAttribute("Addition", "Beispiele", "Addiert zwei Int32 Werte.");
        }

        private NodeVisualization nodeVisualization;
        private ViewPortExtension viewPortExtension;
        private ControlPanelExtension controlPanelExtension;

        private TextBox textBox;
        private Animation demoAnimation;

        // Felder fr Eingnge
        [InPortInfo("a", "Wert A")]
        public int valueA = 0;

        [InPortInfo("b", "Wert B")]
        public int valueB = 0;

        // Felder fr Ausgnge
        [OutPortInfo("c", "Ergebnis", typeof(int))]
        public OutPortHandler ResultUpdate;

        // Datenfelder fr Persistenz
        private int result = 0;

        /// <summary>
        /// Initialisiert eine <see cref="NSampleNode"/>-Instanz.
        /// </summary>
        public NSampleNode()
        {
            // Offene Eingnge mit einfachen Datentypen persistieren
            AttachExtension(new InPortPersistenceExt());

            // Initialsierungen
        }

        /// <summary>
        /// Wird aufgerufen, bevor versucht wird eine Knotenerweiterung zu benutzen.
        /// Diese Methode muss von einem Konten berschrieben werden,
        /// um statische Erweiterungen zu initialisieren.
        /// </summary>
        /// <param name="id">Die ID der Knotenerweiterung, die benutzt werden soll.</param>
        protected override void BeforeUseExtension(string id)
        {
            switch (id)
            {
                case NodeVisualization.EXT_ID:
                    InitVisualization();
                    break;
                case ControlPanelExtension.EXT_ID:
                    InitControlPanel();
                    break;
            }
        }

        /// <summary>
        /// Wird aufgerufen, bevor versucht wird eine Knotenerweiterung zu benutzen.
        /// Diese Methode sollte von einem Konten berschrieben werden,
        /// um statische Erweiterungen zu initialisieren.
        /// </summary>
        /// <param name="type">Die Erweiterungs-Klasse welche benutzt werden soll.</param>
        protected override void BeforeUseExtension(Type type)
        {
            if (type == typeof(ViewPortExtension))
            {
                InitViewPort();
            }
        }

        /// <summary>
        /// Initialsiert die Knoten-Visualisierung.
        /// </summary>
        private void InitVisualization()
        {
            if (nodeVisualization != null) return;

            nodeVisualization = new NodeVisualization();
            nodeVisualization.ClientHeight = 32;
            nodeVisualization.ClientWidth = 64;
            nodeVisualization.CustomAreaPaint += new PaintEventHandler(CustomAreaPaint);


            GetInPortById("a").AttachExtension(
                new PortVisualization(PortOrientation.TOP, 2));
            GetInPortById("b").AttachExtension(
                new PortVisualization(PortOrientation.TOP, 1));
            GetOutPortById("c").AttachExtension(
                new PortVisualization(PortOrientation.BOTTOM, 1));

            AttachExtension(nodeVisualization);
        }

        /// <summary>
        /// Initialisiert die Knoten-Steuerung.
        /// </summary>
        private void InitControlPanel()
        {
            if (controlPanelExtension != null) return;

            controlPanelExtension = new ControlPanelExtension();
            demoAnimation = new Animation();
            demoAnimation.Dock = DockStyle.Fill;
            controlPanelExtension.Control = demoAnimation;
            controlPanelExtension.Show += delegate(object sender, EventArgs e) { demoAnimation.AnimationEnabled = true; };
            controlPanelExtension.Hide += delegate(object sender, EventArgs e) { demoAnimation.AnimationEnabled = false; };
            AttachExtension(controlPanelExtension);
        }

        /// <summary>
        /// Initialisiert den ViewPort.
        /// </summary>
        private void InitViewPort()
        {
            if (viewPortExtension != null) return;

            viewPortExtension = new ViewPortExtension("DemoView", "Additionsergebnisse");

            textBox = new TextBox();
            textBox.Dock = DockStyle.Fill;
            textBox.Multiline = true;
            textBox.ScrollBars = ScrollBars.Both;
            textBox.WordWrap = false;
            viewPortExtension.ViewPortControl = textBox;

            AttachExtension(viewPortExtension);
        }

        /// <summary>
        /// Wird ausgefhrt wenn das Ereignis <see cref="NodeVisualization.CustomAreaPaint"/>
        /// ausgelst wird und zeichnet den benutzerdefinierten Bereich des Knotens.
        /// </summary>
        /// <param name="sender">Das Objekt, welches das Ereignis aufgerufen hat.</param>
        /// <param name="e">Die Ereignisparameter.</param>
        private void CustomAreaPaint(object sender, PaintEventArgs e)
        {
            RectangleF r = nodeVisualization.ClientRectangle;
            Random rand = new Random(DateTime.Now.Millisecond);

            Color c1 = Color.FromArgb(rand.Next(255), rand.Next(255), rand.Next(255));
            Color c2 = Color.FromArgb(rand.Next(255), rand.Next(255), rand.Next(255));

            LinearGradientBrush b = new LinearGradientBrush(
                r.Location,
                new Point((int)(r.Left + r.Width), (int)(r.Top + r.Height)),
                c1, c2);
            e.Graphics.FillRectangle(b, nodeVisualization.ClientRectangle);
            b.Dispose();
        }

        /// <summary>
        /// Fhrt die Knoten-Operation durch.
        /// Dabei knnen die Ausgnge mit neuen Tokens belegt werden.
        /// </summary>
        public override void Work()
        {
            // Ergebnis berechnen
            result = valueA + valueB;

            // Ergebnis auf Ausgang legen
            ResultUpdate(new Token(result));

            // ViewPort aktualisieren
            if (viewPortExtension != null)
            {
                ThreadHelper.SaveInvoke(textBox,
                    delegate(string text)
                    {
                        textBox.Text += text;
                    },
                    "Ergebnis = " + result.ToString() + Environment.NewLine);
            }

            // Benutzerdefinierten Bereich des Knotens neu zeichnen
            if (nodeVisualization != null)
            {
                nodeVisualization.RepaintCustomArea();
            }
        }

        /// <summary>
        /// Wird aufgerufen wenn der Graph gespeichert wird.
        /// </summary>
        /// <param name="pw">Der <see cref="IpropertyWriter"/> mit dem benutzerdefinierte
        /// Eigenschaften des Knotens gespeichert werden knnen.</param>
        /// <seealso cref="Node.OnStore(IPropertyWriter)"/>
        public override void OnStore(IPropertyWriter pw)
        {
            base.OnStore(pw);

            // Weitere Zustande speichern
        }

        /// <summary>
        /// Wird aufgerufen wenn der Graph rekonstruiert wird.
        /// </summary>
        /// <param name="reader">Der <see cref="IpropertyWriter"/> mit dem benutzerdefinierte
        /// Eigenschaften des Knotens gespeichert werden knnen.</param>
        /// <seealso cref="Node.OnRestore(IPropertyReader)"/>
        public override void OnRestore(IPropertyReader pr)
        {
            base.OnRestore(pr);

            // Weitere Zustnde wiederherstellen
        }
    }
}
